#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Program steady_SDPM_NASATM84367 calculates the steady pressure
distributions around the steady swept wing tested by Lockman and Lee
Seegmiller in NASA TM 84367, using the Source and Doublet Panel Method.

This code is part of the SDPMflut Python distribution.
Copyright (C) 2025 Grigorios Dimitriadis

This program is free software: you can redistribute it and/or modify it 
under the terms of the GNU General Public License as published by the 
Free Software Foundation, either version 3 of the License, or (at your 
option) any later version.
 
This program is distributed in the hope that it will be useful, but 
WITHOUT ANY WARRANTY; without even the implied warranty of 
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General 
Public License for more details.

You should have received a copy of the GNU General Public License along 
with this program. If not, see <https://www.gnu.org/licenses/>.
"""

# Input installation directory
install_dir=r'/Users/mbgssgd3/Documents/Python/SDPMflut_v0.72/Common/'
# install_dir=r"C:\Users\Username\Documents\Python\SDPMflut\Common"  # Windows example
# Import libraries and packages
import numpy as np
import matplotlib.pyplot as plt
import scipy.io
import sys
sys.path.append(install_dir)
from SDPMgeometry import SDPMgeometry_trap_fun
import SDPMcalcs
# Create SDPM data types
tp_trap, tp_body, _=SDPMcalcs.SDPMdtypes()

# Run data from wind tunnel tests of wing. Source:
# The forces and pressure distributions at subsonic speeds on a plane wing
# having 45o of sweepback, an aspect ratio of 3, and a taper ratio of 0.5,
# C. D. Kolbe and F. W. Boltz, NACA RM A51G31, 1951.
# Load experimental pressure data from .mat file
mat = scipy.io.loadmat("dataNACARMA51G31.mat")
# Total number of runs
nruns=mat['data'][0].size
# Select angle of attack values to be simulated
alpha0vec=np.array([-2,0,2,4,6,8,10,12,14,16,18])*np.pi/180.0

# Choose order of pressure coefficient equation
cp_order=2 # 1 for linear and 2 for second order

# Set value of mean (or steady) angle of sideslip
beta0=0.0*np.pi/180.0 # Angle of sideslip in rad

# Number of bodies
nbody=1
# Initialize body struct array
body=np.zeros(nbody,dtype=tp_body)

# Input first body
ibody=0         # Index of body
name='wing'     # Name of body
# Choose numbers of panels for this wing and its wake
nhalf=10        # Number of spanwise panels per half-wing.
m=20            # Number of chordwise panels
nchords=10      # Set length of wake in chord lengths
# Calculate number of chordwise wake rings
mw=m*nchords 
# Set number of trapezoidal sections for this wing
ntrap=1;
# Initialize trapezoidal section struct array
trap=np.zeros(ntrap,dtype=tp_trap)

# Input wing geometry
bhalf=46.67*0.0254                    # Span in m of half-wing
c0=41.47*0.0254                       # Root chord in m
lamda=0.5                       # Taper ratio
LamdaLE=48.54*np.pi/180.0;       # Sweep at leading edge in rad
roottwist=0.0*np.pi/180.0       # Twist angle at the root in rad
tiptwist=0.0*np.pi/180.0        # Twist angle at the root in rad
dihedral=0.0*np.pi/180.0        # Dihedral angle in rad
twistcent=0.0                   # Chordwise axis around which twist is defined
# Calculate mean aerodynamic chord
cbarbar=2.0/3.0*c0*(lamda**2+lamda+1)/(lamda+1)
# Spanwise potiion of mean aerodynamic chord
ybarbar=bhalf/(lamda-1)*(cbarbar/c0-1.0)
# Chordwise position of leading edge of mean aerodynamic chord
xbarbar=np.tan(LamdaLE)*ybarbar
# Chordwise distance of root leading edge from previous
# trapezoidal section's tip leading edge
xledist=0.0  
# Set airfoil name (must be the filename of a function in the Common directory)
airfoil='NACA64A010_45'
# Set airfoil parameters
teclosed=1                  # 1 for closed trailing edge, 0 otherwise
# Assemble airfoil parameter values
airfoil_params=np.array([teclosed, 0.0])
# Arrange all data into trapezoidal sections
trap[0]=np.array([(c0,xledist,bhalf,lamda,LamdaLE,roottwist,tiptwist,twistcent,dihedral,airfoil,airfoil_params,airfoil,airfoil_params)],dtype=tp_trap)

# Point around which to calculate moments
xf0=xbarbar+cbarbar/4.0
yf0=0.0
zf0=0.0

# Calculate panel aspect ratio
panelAR=(c0/m)/(bhalf/nhalf)
if panelAR < 0.1:
    sys.exit('Panel aspect ratio too low. Increase n or decrease m.')

# Minimum number of spanwise panels per trapezoidal section
nmin=3
# Chordwise panel distribution: 1 constant, 2 denser at the leading edge
linchord=0
# Spanwise panel distribution: 1 constant, 2 denser at the wing tip(s)
linspan=0
# Define root leading edge
lexyz=np.array([0, 0, 0])
# Define roll, pitch and yaw angles
rollpitchyaw=np.array([0, 0, 0])*np.pi/180;
# Define roll, pitch and yaw centre (x,y,z position of rotation centre)
rollpitchyaw_cent=np.array([0, 0, 0]);
# Input body description
mirroredwing=2 # If mirroredwing=-1: a left half-wing will be created
                # If mirroredwing=1: a right half-wing will be created
                # If mirroredwing=2: two mirrored half-wings will be created.
# dir_tau is the direction in which the unit tangent vector  for this wing
# (tauxx, tauxy, tauxz) has a zero component
dir_tau=2
# Calculate vertices of wing panels
body=SDPMgeometry_trap_fun(body,ibody,m,mw,nhalf,mirroredwing,linchord,linspan,trap,name,dir_tau,rollpitchyaw,rollpitchyaw_cent,lexyz,nmin)

# Plot all bodies
fig, axx = plt.subplots(subplot_kw={"projection": "3d"})
for i in range (0,len(body)):
    axx.plot_surface(body['Xp0'][i], body['Yp0'][i], body['Zp0'][i])
# End for
axx.set_proj_type('ortho')  # FOV = 0 deg
axx.axis('equal')
axx.set_xlabel("$x$", labelpad=10)
axx.set_ylabel("$y$", labelpad=10)
axx.set_zlabel("$z$", labelpad=-1)
axx.view_init(26, -120)
plt.show()

# Assemble the indices of the body panels, spanwise body panels, wake
# panels etc. for all bodies.
allbodies=SDPMcalcs.allbodyindex(body)

print('Calculating flutter solutions for all experimental test cases')
for irun in range (0,nruns):  
    print('Simulating run '+str(irun+1))
    
    # Set Mach number of current run
    Mach=mat['data'][0]['Mach'][irun][0][0]
    # Calculate subsonic compressibility factor
    beta=np.sqrt(1-Mach**2) 
    # Set Reynolds number of current run
    Reynolds=mat['data'][0]['Reynolds'][irun][0][0]
    
    # Calculate steady aerodynamic pressures and loads
    body,allbodies,Aphi,Bphi,Cphi,barUinf,barVinf,barWinf,Cx0,Cy0,Cz0,Cl0,Cm0,Cn0=SDPMcalcs.steadysolve(body,allbodies,cp_order,Mach,beta,alpha0vec,beta0,xf0,yf0,zf0,install_dir)
    
    # Calculate lift coefficient and drag coefficient
    CL0=-Cx0*np.sin(alpha0vec)+Cz0*np.cos(alpha0vec)
    CD0=Cx0*np.cos(alpha0vec)+Cz0*np.sin(alpha0vec)
    
    fig, axx = plt.subplots()
    axx.plot(alpha0vec*180.0/np.pi, CL0, label = "SDPM")
    axx.plot(mat['data'][0]['clalpha'][irun][:,0], mat['data'][0]['clalpha'][irun][:,1], "o",label = "Exp.",)
    axx.set_xlabel("$\\alpha$")
    axx.set_ylabel("$C_L$")
    axx.grid()
    axx.legend(loc="upper left")
    plt.title("$M_{\\infty}=$" +str(Mach) + ", $Re$=" + str(Reynolds))

    fig, axx = plt.subplots()
    axx.plot(Cm0, CL0, label = "SDPM")
    axx.plot(mat['data'][0]['clcm'][irun][:,0], mat['data'][0]['clcm'][irun][:,1], "o",label = "Exp.",)
    axx.set_xlabel("$C_m$")
    axx.set_ylabel("$C_L$")
    axx.grid()
    axx.legend(loc="upper left")
    plt.title("$M_{\\infty}=$" +str(Mach) + ", $Re$=" + str(Reynolds))
    
    fig, axx = plt.subplots()
    axx.plot(CD0, CL0, label = "SDPM")
    axx.plot(CD0+np.min(mat['data'][0]['clcd'][irun][:,0]), CL0, label = "SDPM+CD0")
    axx.plot(mat['data'][0]['clcd'][irun][:,0], mat['data'][0]['clcd'][irun][:,1], "o",label = "Exp.",)
    axx.set_xlabel("$C_D$")
    axx.set_ylabel("$C_L$")
    axx.grid()
    axx.legend(loc="upper left")
    plt.title("$M_{\\infty}=$" +str(Mach) + ", $Re$=" + str(Reynolds))
# End for
